/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#ifndef _mx_arch_h_
#define _mx_arch_h_

#define PAGE_SIZE_FIXED

#include <sys/types.h>
#include <sys/systm.h>   
#include <sys/errno.h>
#include <sys/malloc.h>
#include <sys/queue.h>
#include <machine/limits.h>
#include <IOKit/IOLocks.h>
#include <IOKit/IOLib.h>

#include "mx_auto_config.h"
#include "mx_io_impl.h"
#include "mcp_config.h"
#include "mcp_global.h"
#include "myriexpress.h"

#define MX_PRINT(s) IOLog s
#define MX_WARN(s) do {                 \
  IOLog("mx WARN: ");                  \
  IOLog s;                             \
} while (0)

#define MX_INFO(s) do {                 \
  IOLog("mx INFO: ");                  \
  IOLog s;                             \
} while (0)

#ifndef STAILQ_FOREACH
#define STAILQ_FOREACH(var, head, field)	\
	for((var) = STAILQ_FIRST((head));	\
	    (var); 				\
	    (var) = STAILQ_NEXT((var), field))
#endif /*STAILQ_FOREACH*/
#include "mx_debug.h"
#include "mx_pci.h"

struct mx_instance_state;
typedef void (ether_tx_done_t)(struct mx_instance_state *, uint32_t);
typedef void (ether_rx_done_t)(struct mx_instance_state *, uint32_t, uint32_t, uint32_t, uint32_t);
typedef void (ether_link_change_t)(struct mx_instance_state *);

typedef struct mx_arch_instance_info
{
  char  *csr;                     /* control/status registers */
  void *cpp_class;                /* ptr to c++ class */
  int inuse;                      /* count of users*/
  ether_tx_done_t *ether_tx_done;
  ether_rx_done_t *ether_rx_done_small;
  ether_rx_done_t *ether_rx_done_big;
  ether_link_change_t *ether_link_change;
  struct selinfo  *raw_si;
} mx_arch_instance_info_t;


/*
 * Note that the md and map fields are void * because the real types
 * are defined in c++ header files that the rest of mx would choke on
 * if it had to include them
*/


struct mmapmd_entry
{
  SLIST_ENTRY(mmapmd_entry) entries;
  void *md;       
  void *map;
};

SLIST_HEAD(mmapmd_head, mmapmd_entry); 

typedef struct mx_arch_endpt_info
{
  struct mmapmd_head mmapmd;
  int	 is_64b;
  task_t task;
} mx_arch_endpt_info_t;


#define MX_MTX_STRLEN 64

typedef struct
{
  int cookie;
  SInt32 wake_cnt;
  IOLock *sleep_mtx;
  IOLock *mtx;
} mx_sync_t;


struct mx_page_pin
{
  mcp_dma_addr_t dma;
  uint64_t va;
  void *md;
  struct uio *uio;
};

#define MX_OPTIMIZED_DMA_PAGE_ALLOC 1
#define MX_FACTORIZED_PAGE_PIN 1

#define mx_alloc_copyblock mx_common_alloc_copyblock
#define mx_free_copyblock mx_common_free_copyblock

/* FreeBSD normalizations */

#ifndef MIN
#define MIN(a,b) (((a)<(b))?(a):(b))
#endif

#ifndef MAX
#define MAX(a,b) (((a)>(b))?(a):(b))
#endif

#if MX_DARWIN_XX >= 8
typedef void* mx_kalloc_t;
#define mx_arch_copyin(x, y, z) copyin((mx_uaddr_t)(x), (y), (z))
#define mx_arch_copyout(x, y, z) copyout((x), (mx_uaddr_t)(y), (z))
#define	mx_kgetpid()	((uint32_t)proc_selfpid())
#else
typedef vm_offset_t mx_kalloc_t;
#define mx_arch_copyin(x, y, z) copyin((void *)(x), (y), (z))
#define mx_arch_copyout(x, y, z) copyout((x), (void *)(y), (z))
#define mx_kgetpid()	((uint32_t)((current_proc())->p_pid))
#endif


#define mx_spin(x) IODelay(x)

/* 64-bit byteswapping, in case we need it */
#define mx_hton_u64(x) OSSwapHostToBigInt64(x)
#define mx_ntoh_u64(x) OSSwapBigToHostInt64(x)

/* atomic types and operations */
typedef SInt32 mx_atomic_t;
#define mx_atomic_add(value, ptr)       OSAddAtomic(value, ptr)
#define mx_atomic_subtract(value, ptr)  OSAddAtomic(-value, ptr)
#define mx_atomic_read(ptr)             (*ptr)
#define mx_atomic_set(ptr, value)       *ptr = value

/* spinlocks */
typedef IOSimpleLock *mx_spinlock_t;

#define mx_spin_lock(lock)    IOSimpleLockLock(*lock)
#define mx_spin_unlock(lock)  IOSimpleLockUnlock(*lock)
#define mx_spin_lock_irqsave(lock, flags) \
  do {                                    \
    (*(IOInterruptState *) (&flags)) = IOSimpleLockLockDisableInterrupt(*lock);}\
  while(0)
#define mx_spin_unlock_irqrestore(lock, flags) IOSimpleLockUnlockEnableInterrupt(*lock, *((IOInterruptState *)&flags))

/* mutex operations.  Macros so that the FreeBSD lock
   debugging code can figure out what line of code the mutex was taken
   on */

#define mx_mutex_enter(s)  IOLockLock((s)->mtx)
#define mx_mutex_exit(s)   IOLockUnlock((s)->mtx)

#define mx_reserve_page(x)
#define mx_unreserve_page(x)

#ifndef M_WAITOK
#define M_WAITOK 0
#endif

#define MX_MZERO    M_ZERO
#define MX_WAITOK   M_WAITOK
#define MX_NOWAIT   M_NOWAIT

int  mx_special_create(void);
void mx_special_destroy(void);
void mx_start_timeout_thread(void);
void mx_stop_timeout_thread(void);
void mx_closeall(void);
void mx_cdevsw_remove(void);
void mx_get_memory_size(uint64_t *bytes);

extern int mx_use_count;
extern void **mx_specials;
extern void *mx_ctl[2];
extern int mx_major;
extern int mx_max_minor;

#define mx_get_memory_context() 0

#endif /* _mx_arch_h_ */
